<?php
/**
 * Analytics Event API
 * Tracks user interactions (clicks, navigation, modal views)
 */

require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../includes/analytics-tracker.php';

header('Content-Type: application/json');

// Only accept POST requests
if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

// Get JSON input
$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['event_type'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid request']);
    exit;
}

$eventType = $input['event_type'] ?? '';
$eventData = $input['data'] ?? [];

// Validate event type
$allowedEvents = ['click', 'navigation', 'modal_view', 'page_view'];
if (!in_array($eventType, $allowedEvents)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'message' => 'Invalid event type']);
    exit;
}

// Skip bot requests
if (is_bot_request()) {
    echo json_encode(['success' => false, 'message' => 'Bot request ignored']);
    exit;
}

// Load analytics
$analytics = load_analytics();

// Initialize events array if not exists
if (!isset($analytics['events'])) {
    $analytics['events'] = [];
}

// Create event record
$event = [
    'id' => uniqid('event_', true),
    'timestamp' => time(),
    'date' => date('Y-m-d H:i:s'),
    'event_type' => $eventType,
    'ip' => get_client_ip(),
    'page' => $eventData['page'] ?? $_SERVER['REQUEST_URI'] ?? '/',
    'data' => $eventData
];

// Add event
$analytics['events'][] = $event;

// Keep only last 50000 events to prevent file from growing too large
if (count($analytics['events']) > 50000) {
    $analytics['events'] = array_slice($analytics['events'], -50000);
}

// Update statistics
update_analytics_stats($analytics);

// Save
save_analytics($analytics);

echo json_encode(['success' => true, 'message' => 'Event tracked']);
