<?php
/**
 * Quick Quote API Endpoint
 * Calculates price estimates based on service type and vehicle details
 */

header('Content-Type: application/json');
require_once '../config.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Method not allowed']);
    exit;
}

// Rate limiting
$client_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
if (!check_rate_limit('quote_' . $client_ip, 20, 3600)) {
    http_response_code(429);
    echo json_encode(['success' => false, 'error' => 'Too many requests. Please try again later.']);
    exit;
}

$input = json_decode(file_get_contents('php://input'), true);

if (!$input || !isset($input['service']) || !isset($input['vehicle'])) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Missing required fields'], JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT);
    exit;
}

// Whitelist validation for service types
$allowed_services = ['mot', 'interim', 'full', 'brakes', 'diagnostics', 'aircon', 'clutch', 'tyres', 'cambelt'];
$allowed_vehicles = ['car', 'van'];
$allowed_engines = ['small', 'medium', 'large'];
$allowed_urgency = ['standard', 'urgent', 'emergency'];

$service = sanitize_input($input['service'], 50);
$vehicle = sanitize_input($input['vehicle'], 10);
$engine = isset($input['engine']) ? sanitize_input($input['engine'], 10) : 'medium';
$urgency = isset($input['urgency']) ? sanitize_input($input['urgency'], 20) : 'standard';

// Validate against whitelists
if (!in_array($service, $allowed_services, true)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid service type'], JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT);
    exit;
}

if (!in_array($vehicle, $allowed_vehicles, true)) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid vehicle type'], JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT);
    exit;
}

if (!in_array($engine, $allowed_engines, true)) {
    $engine = 'medium';
}

if (!in_array($urgency, $allowed_urgency, true)) {
    $urgency = 'standard';
}

$pricing = load_json('pricing.json');
$rules = $pricing['estimate_rules'] ?? [];

// Calculate base price
$base_price = null;
$time_estimate = '';

switch ($service) {
    case 'mot':
        $key = $vehicle === 'van' ? 'van' : 'car';
        if (isset($pricing['pricing_bands']['mot'][$key])) {
            $base_price = $pricing['pricing_bands']['mot'][$key]['from'];
        }
        $time_estimate = '45-60 minutes';
        break;
        
    case 'interim':
        if (isset($pricing['pricing_bands']['servicing']['interim'])) {
            $base_price = $pricing['pricing_bands']['servicing']['interim']['from'];
        }
        $time_estimate = '1-2 hours';
        break;
        
    case 'full':
        if (isset($pricing['pricing_bands']['servicing']['full'])) {
            $base_price = $pricing['pricing_bands']['servicing']['full']['from'];
        }
        $time_estimate = '2-4 hours';
        break;
        
    case 'brakes':
        if (isset($pricing['pricing_bands']['brakes']['front_pads'])) {
            $base_price = $pricing['pricing_bands']['brakes']['front_pads']['from'];
        }
        $time_estimate = '1-2 hours';
        break;
        
    case 'diagnostics':
        if (isset($pricing['pricing_bands']['diagnostics']['basic'])) {
            $base_price = $pricing['pricing_bands']['diagnostics']['basic']['from'];
        }
        $time_estimate = '30-60 minutes';
        break;
        
    case 'aircon':
        if (isset($pricing['pricing_bands']['aircon']['regas'])) {
            $base_price = $pricing['pricing_bands']['aircon']['regas']['from'];
        }
        $time_estimate = '1 hour';
        break;
        
    case 'clutch':
        if (isset($pricing['pricing_bands']['clutch']['replacement'])) {
            $base_price = $pricing['pricing_bands']['clutch']['replacement']['from'];
        }
        $time_estimate = '4-6 hours';
        break;
        
    case 'tyres':
        if (isset($pricing['pricing_bands']['tyres']['fitting'])) {
            $base_price = $pricing['pricing_bands']['tyres']['fitting']['from'] * 4; // Estimate for 4 tyres
        }
        $time_estimate = '1 hour';
        break;
        
    case 'cambelt':
        if (isset($pricing['pricing_bands']['cambelt']['replacement'])) {
            $base_price = $pricing['pricing_bands']['cambelt']['replacement']['from'];
        }
        $time_estimate = '3-5 hours';
        break;
}

if ($base_price === null) {
    http_response_code(400);
    echo json_encode(['success' => false, 'error' => 'Invalid service type']);
    exit;
}

// Apply multipliers
$multiplier = 1.0;

// Vehicle multiplier
if (isset($rules['service_multipliers'][$vehicle])) {
    $multiplier *= $rules['service_multipliers'][$vehicle];
}

// Engine size multiplier
if (isset($rules['service_multipliers']['engine_' . $engine])) {
    $multiplier *= $rules['service_multipliers']['engine_' . $engine];
}

// Urgency multiplier
if (isset($rules['urgency_multipliers'][$urgency])) {
    $multiplier *= $rules['urgency_multipliers'][$urgency];
}

$final_price = round($base_price * $multiplier, 2);

// Format price range (add 20% for upper bound)
$upper_price = round($final_price * 1.2, 2);

$price_display = format_currency($final_price);
if ($upper_price > $final_price) {
    $price_display .= ' - ' . format_currency($upper_price);
}

// Safe JSON encoding with proper escaping
echo json_encode([
    'success' => true,
    'price' => $price_display,
    'time' => 'Estimated time: ' . $time_estimate
], JSON_HEX_TAG | JSON_HEX_AMP | JSON_HEX_APOS | JSON_HEX_QUOT);
