# Central Router System Guide

## Overview

The website now uses a centralized routing system with `router.php` and `routes.php` for seamless URL handling, 404s, and page serving.

## How It Works

### 1. Request Flow

```
User Request → .htaccess → router.php → routes.php → Page File
```

1. User visits a URL (e.g., `/services/mot-coventry`)
2. `.htaccess` routes all non-file requests to `router.php`
3. `router.php` matches the URL against routes in `routes.php`
4. If matched, includes the appropriate page file
5. If not matched, shows 404 page

### 2. Route Definition (`routes.php`)

Routes are defined in a simple array structure:

```php
'/services/{slug}' => [
    'file' => 'service-detail.php',
    'title' => 'Service',
    'method' => 'GET',
    'params' => ['slug']
],
```

**Route Pattern:**
- Static: `/services` → exact match
- Dynamic: `/services/{slug}` → captures slug parameter

**Route Config:**
- `file` - PHP file to include
- `title` - Page title (for reference)
- `method` - HTTP methods allowed (GET, POST, GET|POST)
- `params` - Parameter names for dynamic routes
- `api` - Set to `true` for API routes (no header/footer)

### 3. Router (`router.php`)

The router:
- Parses the request URL
- Matches against route patterns
- Extracts parameters from dynamic routes
- Validates HTTP method
- Includes the appropriate file
- Handles 404s automatically

## Adding New Routes

### Simple Page Route

```php
'/new-page' => [
    'file' => 'new-page.php',
    'title' => 'New Page',
    'method' => 'GET'
],
```

### Dynamic Route with Parameters

```php
'/blog/{slug}' => [
    'file' => 'blog-post.php',
    'title' => 'Blog Post',
    'method' => 'GET',
    'params' => ['slug']
],
```

### POST Route

```php
'/contact' => [
    'file' => 'contact.php',
    'title' => 'Contact',
    'method' => 'GET|POST'
],
```

### API Route

```php
'/api/data' => [
    'file' => 'api/data.php',
    'title' => 'Data API',
    'method' => 'GET',
    'api' => true  // No header/footer
],
```

## Accessing Route Parameters

In your page file, access parameters via `$_GET`:

```php
// In service-detail.php
$slug = $_GET['slug'] ?? '';

// In blog-post.php
$post_slug = $_GET['slug'] ?? '';
```

The router automatically sets these from the URL pattern.

## 404 Handling

404s are handled automatically:

1. If no route matches → 404
2. If route file doesn't exist → 404
3. If HTTP method not allowed → tries next route, then 404

The 404 page is shown seamlessly without redirects.

## Current Routes

### Main Pages
- `/` → Homepage
- `/services` → Services hub
- `/pricing` → Pricing
- `/about` → About
- `/contact` → Contact (GET|POST)
- `/reviews` → Reviews
- `/faq` → FAQ
- `/sitemap` → HTML Sitemap

### Dynamic Routes
- `/services/{slug}` → Service detail pages

### API Routes
- `/api/quote` → Quote calculator (POST)
- `/sitemap.xml` → XML sitemap (GET)

### Policy Pages
- `/privacy-policy` → Privacy Policy
- `/cookie-policy` → Cookie Policy
- `/terms` → Terms of Service

## Benefits

### ✅ Centralized Routing
- All routes in one file (`routes.php`)
- Easy to see all available URLs
- Simple to add/modify routes

### ✅ Automatic 404s
- No route matching → 404
- Missing file → 404
- No manual 404 handling needed

### ✅ Clean URLs
- SEO-friendly URLs
- No `.php` extensions
- Works on any domain

### ✅ Parameter Extraction
- Automatic parameter extraction
- Available in `$_GET` for backward compatibility
- Validated and sanitized

### ✅ Method Validation
- HTTP method checking
- GET, POST, or both
- Prevents unauthorized access

## File Structure

```
/
├── router.php          # Main router (entry point)
├── routes.php           # Route definitions
├── config.php           # Configuration
├── index.php            # Homepage
├── services.php         # Services page
├── service-detail.php    # Service detail (uses {slug})
├── api/
│   └── quote.php        # API endpoint
└── 404.php              # 404 error page
```

## Testing Routes

### Test All Routes

Visit `/test-routes.php?key=test123` to see:
- All defined routes
- Current route matching
- Parameter extraction

### Manual Testing

1. **Homepage:** `/`
2. **Services:** `/services`
3. **Service Detail:** `/services/mot-coventry`
4. **Invalid Route:** `/invalid-route` → 404

## Troubleshooting

### Route Not Working?

1. **Check route definition** in `routes.php`
2. **Verify file exists** (route config `file` value)
3. **Check HTTP method** matches request
4. **Check `.htaccess`** is routing to `router.php`

### 404 on Valid Route?

1. **Check route pattern** matches URL exactly
2. **Verify file path** in route config
3. **Check file permissions**
4. **Review router.php** error handling

### Parameters Not Working?

1. **Check `params` array** in route config
2. **Verify parameter names** match usage
3. **Access via `$_GET['param_name']`**

## Advanced Usage

### Multiple Parameters

```php
'/blog/{year}/{month}/{slug}' => [
    'file' => 'blog-post.php',
    'params' => ['year', 'month', 'slug']
],
```

Access: `$_GET['year']`, `$_GET['month']`, `$_GET['slug']`

### Optional Parameters

Use separate routes:

```php
'/blog' => ['file' => 'blog.php'],
'/blog/{slug}' => ['file' => 'blog-post.php', 'params' => ['slug']],
```

### Route Groups (Future)

For complex sites, you could extend the router to support route groups:

```php
'blog' => [
    '/' => ['file' => 'blog.php'],
    '/{slug}' => ['file' => 'blog-post.php', 'params' => ['slug']],
],
```

## Migration Notes

### Old System
- Relied on `.htaccess` rewrites
- Manual 404 handling
- Scattered route logic

### New System
- Centralized routing
- Automatic 404s
- Clean, maintainable code

### Backward Compatibility

All existing pages work as-is:
- Parameters still in `$_GET`
- `$current_slug` still available
- No changes needed to page files

---

**Status:** ✅ Central router system implemented
**Last Updated:** April 2024
