<?php
/**
 * Bot Detection - Don't show modal to search engines or AI crawlers
 */
function isBot() {
    $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? '';
    $userAgentLower = strtolower($userAgent);
    
    // Common bot user agents
    $botPatterns = [
        // Search engines
        'googlebot', 'bingbot', 'slurp', 'duckduckbot', 'baiduspider', 'yandexbot',
        'sogou', 'exabot', 'facebot', 'ia_archiver', 'msnbot', 'ahrefsbot',
        // AI crawlers
        'chatgpt', 'claude', 'anthropic', 'openai', 'gpt', 'perplexity', 'crawl',
        'crawler', 'spider', 'bot', 'scraper', 'fetcher', 'indexer',
        // Social media bots
        'facebookexternalhit', 'twitterbot', 'linkedinbot', 'whatsapp', 'telegram',
        // Other bots
        'curl', 'wget', 'python-requests', 'scrapy', 'headless', 'phantomjs'
    ];
    
    foreach ($botPatterns as $pattern) {
        if (strpos($userAgentLower, $pattern) !== false) {
            return true;
        }
    }
    
    // Check for missing or suspicious user agent
    if (empty($userAgent) || strlen($userAgent) < 10) {
        return true;
    }
    
    // Check if it's a headless browser (common indicators)
    if (strpos($userAgentLower, 'headless') !== false || 
        strpos($userAgentLower, 'phantom') !== false ||
        strpos($userAgentLower, 'selenium') !== false) {
        return true;
    }
    
    return false;
}

// Don't show to bots at all
$isBot = isBot();

// Check if user has dismissed the proposal modal (only for real users)
$proposal_dismissed = false;
$proposal_dismiss_time = 0;

if (!$isBot) {
    $proposal_dismissed = isset($_COOKIE['proposal_modal_dismissed']) && $_COOKIE['proposal_modal_dismissed'] === 'true';
    $proposal_dismiss_time = isset($_COOKIE['proposal_modal_dismiss_time']) ? (int)$_COOKIE['proposal_modal_dismiss_time'] : 0;
}

// Check if 24 hours have passed since dismissal
$dismiss_expired = false;
if ($proposal_dismiss_time > 0) {
    $hours_passed = (time() - $proposal_dismiss_time) / 3600;
    $dismiss_expired = $hours_passed >= 24;
}

// Show modal only to real users (not bots) and if not dismissed or if 24 hours have passed
$show_proposal_modal = !$isBot && (!$proposal_dismissed || $dismiss_expired);
?>

<?php if ($show_proposal_modal): ?>
<!-- Proposal Modal - Hidden from search engines and bots -->
<div id="proposal-modal" 
     class="fixed inset-0 bg-black bg-opacity-50 z-50 hidden items-center justify-center p-4" 
     style="display: none;"
     data-nosnippet="true"
     data-noindex="true"
     aria-hidden="true">
    <div class="bg-white rounded-lg shadow-2xl max-w-2xl w-full max-h-[90vh] overflow-y-auto transform transition-all">
        <!-- Modal Header -->
        <div class="bg-gradient-to-r from-blue-600 to-blue-800 text-white p-6 rounded-t-lg">
            <div class="flex items-center justify-between">
                <div class="flex items-center gap-3">
                    <svg class="w-8 h-8" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"></path>
                    </svg>
                    <h2 class="text-2xl font-bold">Project Proposal</h2>
                </div>
                <button onclick="dismissProposalModal(24)" 
                        class="text-white hover:text-gray-200 transition p-2 hover:bg-white hover:bg-opacity-20 rounded-full"
                        aria-label="Close">
                    <svg class="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"></path>
                    </svg>
                </button>
            </div>
        </div>
        
        <!-- Modal Body -->
        <div class="p-6 md:p-8">
            <div class="space-y-6">
                <!-- Main Message -->
                <div class="text-center">
                    <p class="text-lg text-gray-700 leading-relaxed mb-4">
                        This website was created as a <strong class="text-blue-600">project proposal</strong> demonstrating a fully functional, 
                        SEO-optimized template for a mechanic/garage business.
                    </p>
                </div>
                
                <!-- Features List -->
                <div class="bg-blue-50 rounded-lg p-6 border border-blue-200">
                    <h3 class="text-xl font-bold text-gray-900 mb-4 flex items-center gap-2">
                        <svg class="w-6 h-6 text-blue-600" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m5.618-4.016A11.955 11.955 0 0112 2.944a11.955 11.955 0 01-8.618 3.04A12.02 12.02 0 003 9c0 5.591 3.824 10.29 9 11.622 5.176-1.332 9-6.03 9-11.622 0-1.042-.133-2.052-.382-3.016z"></path>
                        </svg>
                        What's Included
                    </h3>
                    <ul class="space-y-3 text-gray-700">
                        <li class="flex items-start gap-3">
                            <svg class="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                            </svg>
                            <span><strong>Domain for Sale:</strong> This domain is available for purchase</span>
                        </li>
                        <li class="flex items-start gap-3">
                            <svg class="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                            </svg>
                            <span><strong>Complete Template:</strong> Fully customizable template ready for deployment</span>
                        </li>
                        <li class="flex items-start gap-3">
                            <svg class="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                            </svg>
                            <span><strong>SEO Optimized:</strong> Built-in SEO features, sitemap, schema markup</span>
                        </li>
                        <li class="flex items-start gap-3">
                            <svg class="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                            </svg>
                            <span><strong>Mobile Responsive:</strong> Fully responsive design for all devices</span>
                        </li>
                        <li class="flex items-start gap-3">
                            <svg class="w-5 h-5 text-green-500 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                <path fill-rule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clip-rule="evenodd"></path>
                            </svg>
                            <span><strong>Easy Customization:</strong> JSON-driven content, no database required</span>
                        </li>
                    </ul>
                </div>
                
                <!-- Call to Action -->
                <div class="bg-gradient-to-r from-green-50 to-blue-50 rounded-lg p-6 border-2 border-green-200">
                    <h3 class="text-xl font-bold text-gray-900 mb-3 text-center">Interested in This Project?</h3>
                    <p class="text-center text-gray-700 mb-6">
                        Feel free to contact us via WhatsApp to discuss purchasing the domain, 
                        customizing the template, or any questions you may have.
                    </p>
                    
                    <!-- WhatsApp Button -->
                    <div class="text-center">
                        <a href="https://wa.me/447926777005?text=Hi,%20I'm%20interested%20in%20the%20domain%20and%20template%20proposal" 
                           target="_blank"
                           rel="noopener noreferrer"
                           class="inline-flex items-center gap-3 bg-green-500 hover:bg-green-600 text-white px-8 py-4 rounded-lg font-semibold text-lg shadow-lg hover:shadow-xl transition-all transform hover:scale-105">
                            <svg class="w-6 h-6" fill="currentColor" viewBox="0 0 24 24">
                                <path d="M17.472 14.382c-.297-.149-1.758-.867-2.03-.967-.273-.099-.471-.148-.67.15-.197.297-.767.966-.94 1.164-.173.199-.347.223-.644.075-.297-.15-1.255-.463-2.39-1.475-.883-.788-1.48-1.761-1.653-2.059-.173-.297-.018-.458.13-.606.134-.133.298-.347.446-.52.149-.174.198-.298.298-.497.099-.198.05-.371-.025-.52-.075-.149-.669-1.612-.916-2.207-.242-.579-.487-.5-.669-.51-.173-.008-.371-.01-.57-.01-.198 0-.52.074-.792.372-.272.297-1.04 1.016-1.04 2.479 0 1.462 1.065 2.875 1.213 3.074.149.198 2.096 3.2 5.077 4.487.709.306 1.262.489 1.694.625.712.227 1.36.195 1.871.118.571-.085 1.758-.719 2.006-1.413.248-.694.248-1.289.173-1.413-.074-.124-.272-.198-.57-.347m-5.421 7.403h-.004a9.87 9.87 0 01-5.031-1.378l-.361-.214-3.741.982.998-3.648-.235-.374a9.86 9.86 0 01-1.51-5.26c.001-5.45 4.436-9.884 9.888-9.884 2.64 0 5.122 1.03 6.988 2.898a9.825 9.825 0 012.893 6.994c-.003 5.45-4.437 9.884-9.885 9.884m8.413-18.297A11.815 11.815 0 0012.05 0C5.495 0 .16 5.335.157 11.892c0 2.096.547 4.142 1.588 5.945L.057 24l6.305-1.654a11.882 11.882 0 005.683 1.448h.005c6.554 0 11.89-5.335 11.893-11.893a11.821 11.821 0 00-3.48-8.413Z"/>
                            </svg>
                            Contact Us on WhatsApp
                        </a>
                        <p class="text-sm text-gray-600 mt-3">
                            <strong>Phone:</strong> <a href="tel:+447926777005" class="text-blue-600 hover:text-blue-700">+44 7926 777005</a>
                        </p>
                    </div>
                </div>
            </div>
        </div>
        
        <!-- Modal Footer -->
        <div class="bg-gray-50 px-6 py-4 rounded-b-lg border-t border-gray-200 flex items-center justify-between">
            <p class="text-sm text-gray-600">
                This modal can be dismissed for 24 hours
            </p>
            <button onclick="dismissProposalModal(24)" 
                    class="bg-gray-200 hover:bg-gray-300 text-gray-800 px-6 py-2 rounded-lg font-medium transition">
                Dismiss for 24 Hours
            </button>
        </div>
    </div>
</div>

<style>
#proposal-modal {
    animation: fadeIn 0.3s ease-out;
}

#proposal-modal .bg-white {
    animation: slideUp 0.3s ease-out;
}

#proposal-modal.hidden {
    animation: fadeOut 0.3s ease-in;
}

#proposal-modal.hidden .bg-white {
    animation: slideDown 0.3s ease-in;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

@keyframes fadeOut {
    from { opacity: 1; }
    to { opacity: 0; }
}

@keyframes slideUp {
    from {
        transform: translateY(20px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

@keyframes slideDown {
    from {
        transform: translateY(0);
        opacity: 1;
    }
    to {
        transform: translateY(20px);
        opacity: 0;
    }
}
</style>

<script>
// Check if cookie says modal should be dismissed
function isModalDismissed() {
    const dismissed = getCookie('proposal_modal_dismissed');
    const dismissTime = parseInt(getCookie('proposal_modal_dismiss_time') || '0');
    
    if (dismissed === 'true' && dismissTime > 0) {
        // Check if 24 hours have passed
        const currentTime = Math.floor(Date.now() / 1000);
        const hoursPassed = (currentTime - dismissTime) / 3600;
        
        if (hoursPassed < 24) {
            return true; // Still dismissed
        }
    }
    
    return false; // Not dismissed or expired
}

// Helper function to get cookie
function getCookie(name) {
    const value = `; ${document.cookie}`;
    const parts = value.split(`; ${name}=`);
    if (parts.length === 2) return parts.pop().split(';').shift();
    return '';
}

// Track modal state
let modalShown = false;
let clickCount = 0;

// Don't show if dismissed by cookie
if (isModalDismissed()) {
    // Remove modal from DOM if dismissed
    const modal = document.getElementById('proposal-modal');
    if (modal) {
        modal.remove();
    }
} else {
    // Function to show modal
    function showProposalModal() {
        // Check cookie again before showing
        if (isModalDismissed()) {
            return;
        }
        
        const modal = document.getElementById('proposal-modal');
        if (modal && !modalShown) {
            modalShown = true;
            modal.classList.remove('hidden');
            modal.style.display = 'flex';
            document.body.style.overflow = 'hidden'; // Prevent background scrolling
            
            // Track modal view
            trackAnalyticsEvent('modal_view', {
                modal_type: 'proposal',
                page: window.location.pathname
            });
        }
    }
    
    // Remove page view tracking (not needed)

    // Track clicks and show modal ONLY after 2-3 clicks (no auto-show, no scroll trigger)
    document.addEventListener('click', function(e) {
        // Don't count clicks on the modal itself or its buttons
        if (e.target.closest('#proposal-modal')) {
            return;
        }
        
        // Don't show if dismissed
        if (isModalDismissed()) {
            return;
        }
        
        // Don't count clicks on navigation links (they navigate away)
        const link = e.target.closest('a');
        if (link && link.getAttribute('href')) {
            const href = link.getAttribute('href');
            // Only skip external links and internal navigation (not anchor links)
            if (href.startsWith('http') || (href.startsWith('/') && !href.startsWith('#'))) {
                return; // Don't count navigation clicks
            }
        }
        
        clickCount++;
        
        // Show modal after 2 clicks if not already shown
        if (clickCount >= 2 && !modalShown) {
            showProposalModal();
        }
    });
}

function dismissProposalModal(hours) {
    const modal = document.getElementById('proposal-modal');
    if (modal) {
        modal.classList.add('hidden');
        setTimeout(function() {
            modal.style.display = 'none';
            document.body.style.overflow = ''; // Restore scrolling
        }, 300);
        
        // Set cookie to dismiss for specified hours
        const expires = new Date();
        expires.setTime(expires.getTime() + (hours * 60 * 60 * 1000));
        document.cookie = 'proposal_modal_dismissed=true;expires=' + expires.toUTCString() + ';path=/;SameSite=Lax';
        document.cookie = 'proposal_modal_dismiss_time=' + Math.floor(Date.now() / 1000) + ';expires=' + expires.toUTCString() + ';path=/;SameSite=Lax';
    }
}

// Close modal when clicking outside
var proposalModal = document.getElementById('proposal-modal');
if (proposalModal) {
    proposalModal.addEventListener('click', function(e) {
        if (e.target === this) {
            dismissProposalModal(24);
        }
    });
}

// Close modal with Escape key
document.addEventListener('keydown', function(e) {
    if (e.key === 'Escape') {
        const modal = document.getElementById('proposal-modal');
        if (modal && modal.style.display !== 'none') {
            dismissProposalModal(24);
        }
    }
});
</script>
<?php endif; ?>
