<?php
/**
 * Manual Sitemap Ping Script
 * Use this to manually notify search engines about sitemap updates
 * Access at: /ping-sitemap?key=your_secret_key
 * 
 * Note: Google and Bing have deprecated their ping endpoints.
 * This script now uses IndexNow protocol (supported by Bing, Yandex, etc.)
 * For Google, use Google Search Console to submit sitemaps.
 */

require_once 'config.php';

// Security: Require a secret key to prevent abuse
$secret_key = 'change_this_secret_key_in_production';
$provided_key = $_GET['key'] ?? '';

if ($provided_key !== $secret_key) {
    http_response_code(403);
    header('Content-Type: application/json');
    echo json_encode([
        'error' => 'Access denied. Invalid key.',
        'message' => 'Please provide a valid key parameter: /ping-sitemap?key=your_secret_key'
    ], JSON_PRETTY_PRINT);
    exit;
}

header('Content-Type: application/json');

$sitemap_url = SITE_URL . '/sitemap.xml';
$results = [];
$info = [];

// Note: Google and Bing deprecated their ping endpoints
$info[] = 'Note: Google and Bing have deprecated their sitemap ping endpoints.';
$info[] = 'For Google: Submit sitemap via Google Search Console.';
$info[] = 'For Bing: Use IndexNow protocol (implemented below).';

// IndexNow Protocol (supported by Bing, Yandex, Seznam.cz, Naver, etc.)
// This is the modern way to notify search engines
$indexnow_key = substr(md5(SITE_URL), 0, 32); // Generate a key based on domain
$indexnow_endpoints = [
    'Bing IndexNow' => 'https://api.indexnow.org/indexnow',
    'Yandex IndexNow' => 'https://yandex.com/indexnow',
];

// Try IndexNow protocol
foreach ($indexnow_endpoints as $engine => $endpoint) {
    $result = ['engine' => $engine, 'method' => 'IndexNow', 'status' => 'pending'];
    
    // IndexNow requires POST with JSON
    $indexnow_data = [
        'host' => parse_url(SITE_URL, PHP_URL_HOST),
        'key' => $indexnow_key,
        'urlList' => [$sitemap_url]
    ];
    
    if (function_exists('curl_init')) {
        $ch = curl_init($endpoint);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($indexnow_data));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'User-Agent: IndexNow/1.0'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 10);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            $result['status'] = 'error';
            $result['message'] = $error;
        } elseif ($http_code == 200 || $http_code == 202) {
            $result['status'] = 'success';
            $result['http_code'] = $http_code;
        } else {
            $result['status'] = 'failed';
            $result['http_code'] = $http_code;
            $result['message'] = 'Unexpected response code';
        }
    } else {
        $result['status'] = 'error';
        $result['message'] = 'cURL not available';
    }
    
    $results[] = $result;
}

// Legacy ping endpoints (may not work, but we'll try)
$legacy_endpoints = [
    'Google (Deprecated)' => 'https://www.google.com/ping?sitemap=' . urlencode($sitemap_url),
    'Bing (Deprecated)' => 'https://www.bing.com/ping?sitemap=' . urlencode($sitemap_url),
];

// Try legacy endpoints (for reference, likely won't work)
foreach ($legacy_endpoints as $engine => $ping_url) {
    $result = ['engine' => $engine, 'method' => 'Legacy Ping', 'url' => $ping_url, 'status' => 'pending'];
    
    if (function_exists('curl_init')) {
        $ch = curl_init($ping_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_TIMEOUT, 5);
        curl_setopt($ch, CURLOPT_CONNECTTIMEOUT, 5);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Sitemap-Ping/1.0');
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($ch, CURLOPT_NOBODY, true); // HEAD request only
        curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $error = curl_error($ch);
        curl_close($ch);
        
        if ($error) {
            $result['status'] = 'error';
            $result['message'] = $error;
        } elseif ($http_code >= 200 && $http_code < 300) {
            $result['status'] = 'success';
            $result['http_code'] = $http_code;
        } else {
            $result['status'] = 'deprecated';
            $result['http_code'] = $http_code;
            $result['message'] = 'Endpoint deprecated - use Search Console/IndexNow instead';
        }
    } else {
        $result['status'] = 'error';
        $result['message'] = 'cURL not available';
    }
    
    $results[] = $result;
}

// Generate response
$response = [
    'sitemap_url' => $sitemap_url,
    'timestamp' => date('Y-m-d H:i:s'),
    'info' => $info,
    'recommendations' => [
        'Google' => 'Submit sitemap manually via Google Search Console: https://search.google.com/search-console',
        'Bing' => 'IndexNow protocol used above, or submit via Bing Webmaster Tools',
        'Yandex' => 'IndexNow protocol used above, or submit via Yandex Webmaster',
        'robots.txt' => 'Ensure your robots.txt includes: Sitemap: ' . $sitemap_url
    ],
    'results' => $results
];

echo json_encode($response, JSON_PRETTY_PRINT);
