<?php
/**
 * Central Router - Simplified version
 * Handles all URL routing and page serving
 */

// Error logging
error_reporting(E_ALL);
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/errors.log');

// Load config
$config_file = __DIR__ . '/config.php';
if (!file_exists($config_file)) {
    error_log("Config file not found: {$config_file}");
    http_response_code(500);
    die('Configuration file not found.');
}
require_once $config_file;

// Load analytics tracker (for tracking visits)
$analytics_tracker = __DIR__ . '/includes/analytics-tracker.php';
if (file_exists($analytics_tracker)) {
    require_once $analytics_tracker;
}

// Get URL from request
$request_uri = $_SERVER['REQUEST_URI'] ?? '/';
$request_path = parse_url($request_uri, PHP_URL_PATH);

// Remove base path if in subdirectory
$script_name = dirname($_SERVER['SCRIPT_NAME'] ?? '');
if ($script_name !== '/' && $script_name !== '\\' && $script_name !== '.' && $script_name !== '') {
    $request_path = str_replace($script_name, '', $request_path);
}

$request_path = '/' . trim($request_path, '/');
if ($request_path === '//' || $request_path === '') {
    $request_path = '/';
}

// Handle static files early
$static_files = ['/favicon.ico', '/robots.txt'];
if (in_array($request_path, $static_files)) {
    http_response_code(404);
    exit;
}

// Load routes
$routes_file = __DIR__ . '/routes.php';
if (!file_exists($routes_file)) {
    error_log("Routes file not found: {$routes_file}");
    http_response_code(500);
    die('Routes file not found.');
}

$routes = require $routes_file;
if (!is_array($routes)) {
    error_log("Invalid routes configuration");
    http_response_code(500);
    die('Invalid routes configuration.');
}

// Find matching route
$matched_route = null;
$route_params = [];

foreach ($routes as $route_pattern => $route_config) {
    // Convert route pattern to regex
    $pattern = '#^' . preg_replace('/\{([^}]+)\}/', '([^/]+)', $route_pattern) . '$#';
    
    if (preg_match($pattern, $request_path, $matches)) {
        // Check HTTP method
        $allowed_methods = explode('|', $route_config['method'] ?? 'GET');
        if (!in_array($_SERVER['REQUEST_METHOD'] ?? 'GET', $allowed_methods)) {
            continue;
        }
        
        // Extract parameters
        if (isset($route_config['params']) && count($matches) > 1) {
            $param_names = $route_config['params'];
            array_shift($matches);
            foreach ($param_names as $index => $param_name) {
                if (isset($matches[$index])) {
                    $route_params[$param_name] = $matches[$index];
                    $_GET[$param_name] = $matches[$index];
                }
            }
        }
        
        $matched_route = $route_config;
        break;
    }
}

// Handle 404
if (!$matched_route) {
    http_response_code(404);
    $current_slug = '404';
    $GLOBALS['current_slug'] = $current_slug;
    
    // API request?
    if (strpos($_SERVER['HTTP_ACCEPT'] ?? '', 'application/json') !== false || 
        strpos($request_path, '/api/') === 0) {
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Not Found', 'status' => 404]);
        exit;
    }
    
    // Show 404 page
    $not_found_file = __DIR__ . '/views/404.php';
    if (file_exists($not_found_file)) {
        include $not_found_file;
    } else {
        die('404 - Page Not Found');
    }
    exit;
}

// Set current slug
$current_slug = trim($request_path, '/') ?: 'home';
$GLOBALS['current_slug'] = $current_slug;

// API route?
if (isset($matched_route['api']) && $matched_route['api']) {
    $route_file = __DIR__ . '/' . ltrim($matched_route['file'], '/');
    if (file_exists($route_file)) {
        require $route_file;
    } else {
        error_log("API route file not found: {$route_file}");
        http_response_code(404);
        header('Content-Type: application/json');
        echo json_encode(['error' => 'Route file not found', 'status' => 404]);
    }
    exit;
}

// Track visit (before rendering page, but after route matching)
if (function_exists('track_visit')) {
    track_visit($request_path);
}

// Auto-ping sitemap every 7 days (on any user visit)
if (function_exists('ping_search_engines')) {
    $sitemap_url = SITE_URL . '/sitemap.xml';
    ping_search_engines($sitemap_url);
}

// Regular page route
$route_file = __DIR__ . '/' . ltrim($matched_route['file'], '/');
if (file_exists($route_file)) {
    require $route_file;
} else {
    error_log("Route file not found: {$route_file}");
    http_response_code(404);
    $current_slug = '404';
    $GLOBALS['current_slug'] = $current_slug;
    $not_found_file = __DIR__ . '/views/404.php';
    if (file_exists($not_found_file)) {
        include $not_found_file;
    } else {
        die('404 - Page Not Found');
    }
}
