<?php
/**
 * Dynamic XML Sitemap Generator
 * Generates sitemap.xml for search engines and automatically pings them
 */
require_once 'config.php';

header('Content-Type: application/xml; charset=utf-8');

// Ping search engines about sitemap (non-blocking, rate-limited)
// Only ping if function exists and we're in production
if (function_exists('ping_search_engines')) {
    $sitemap_url = SITE_URL . '/sitemap.xml';
    // Ping in background (don't wait for response)
    if (function_exists('fastcgi_finish_request')) {
        // FastCGI: finish response first, then ping
        register_shutdown_function(function() use ($sitemap_url) {
            if (function_exists('ping_search_engines')) {
                @ping_search_engines($sitemap_url);
            }
        });
    } else {
        // Regular PHP: ping asynchronously (suppress errors)
        @ping_search_engines($sitemap_url);
    }
}

$urls = [];

// Homepage
$urls[] = [
    'loc' => SITE_URL . '/',
    'lastmod' => date('Y-m-d'),
    'changefreq' => 'weekly',
    'priority' => '1.0'
];

// Main pages
$main_pages = [
    'services' => 'weekly',
    'pricing' => 'monthly',
    'about' => 'monthly',
    'reviews' => 'weekly',
    'faq' => 'monthly',
    'contact' => 'monthly'
];

foreach ($main_pages as $page => $freq) {
    $urls[] = [
        'loc' => SITE_URL . '/' . $page,
        'lastmod' => date('Y-m-d'),
        'changefreq' => $freq,
        'priority' => '0.8'
    ];
}

// Service detail pages
if ($services_data && isset($services_data['services'])) {
    foreach ($services_data['services'] as $service) {
        $urls[] = [
            'loc' => SITE_URL . '/services/' . $service['slug'],
            'lastmod' => date('Y-m-d'),
            'changefreq' => 'monthly',
            'priority' => '0.9'
        ];
    }
}

// Policy pages
$policy_pages = ['privacy-policy', 'cookie-policy', 'terms', 'sitemap'];
foreach ($policy_pages as $page) {
    $urls[] = [
        'loc' => SITE_URL . '/' . $page,
        'lastmod' => date('Y-m-d'),
        'changefreq' => 'yearly',
        'priority' => '0.3'
    ];
}

// Output XML
echo '<?xml version="1.0" encoding="UTF-8"?>' . "\n";
echo '<urlset xmlns="http://www.sitemaps.org/schemas/sitemap/0.9">' . "\n";

foreach ($urls as $url) {
    echo "  <url>\n";
    echo "    <loc>" . htmlspecialchars($url['loc']) . "</loc>\n";
    echo "    <lastmod>" . $url['lastmod'] . "</lastmod>\n";
    echo "    <changefreq>" . $url['changefreq'] . "</changefreq>\n";
    echo "    <priority>" . $url['priority'] . "</priority>\n";
    echo "  </url>\n";
}

echo '</urlset>';
